package com.hero.objects.talents;

import java.util.ArrayList;
import java.util.Iterator;

import org.jdom.Element;

import com.hero.HeroDesigner;
import com.hero.objects.GenericObject;
import com.hero.objects.characteristics.Characteristic;
import com.hero.objects.skills.CharacteristicChoice;
import com.hero.ui.dialog.CombatSenseDialog;
import com.hero.ui.dialog.GenericDialog;
import com.hero.util.Constants;
import com.hero.util.Rounder;
import com.hero.util.XMLUtility;

/**
 * Copyright (c) 2000 - 2005, CompNet Design, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, is prohibited unless the following conditions are met: 1.
 * Express written consent of CompNet Design, Inc. is obtained by the developer.
 * 2. Redistributions must retain this copyright notice. THIS SOFTWARE IS
 * PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * @author CompNet Design, Inc.
 * @version $Revision$
 */

public class CombatSense extends Talent {

	private static String xmlID = "COMBAT_SENSE";

	protected int characteristic;

	protected ArrayList<CharacteristicChoice> characteristicChoices;

	public CombatSense(Element root) {
		super(root, CombatSense.xmlID);
	}

	/**
	 * The Characteristic that the roll is based off of.
	 * 
	 * @see com.hero.util.Constants
	 * @return
	 */
	public int getCharacteristic() {
		return characteristic;
	}

	/**
	 * Returns a Vector of CharacteristicChoice objects representing the options
	 * for the roll.
	 * 
	 * @see com.hero.objects.skills.CharacteristicChoice
	 * @return
	 */
	public ArrayList<CharacteristicChoice> getCharacteristicChoices() {
		return characteristicChoices;
	}

	@Override
	public String getColumn2Output() {
		String ret = getAlias();
		if ((getName() != null) && (getName().trim().length() > 0)) {
			ret = "<i>" + getName() + ":</i>  " + ret;
		}
		if ((getInput() != null) && (getInput().trim().length() > 0)) {
			ret += ":  " + getInput();
		}
		if (!HeroDesigner.getInstance().getPrefs().useWG()
				&& (getCharacteristicChoices() != null)
				&& (getCharacteristicChoices().size() > 1)
				&& !(getCharacteristic() == Constants.GENERAL)) {
			ret += " ("
					+ Constants.getCharacteristicString(getCharacteristic())
					+ "-based)";
		}
		String adderString = getAdderString();
		if (adderString.trim().length() > 0) {
			ret += " (";
			ret += adderString;
			ret += ")";
		}
		ret += getModifierString();
		ret += " " + getRoll();
		if ((getEndUsage() > 0)
				&& (GenericObject.findObjectByID(HeroDesigner.getActiveHero()
						.getPowers(), "ENDURANCERESERVE") != null)
				&& (GenericObject.findObjectByID(getAllAssignedModifiers(),
						"ENDRESERVEOREND") == null)
				&& !HeroDesigner.getInstance().getPrefs().useWG()) {
			if (getUseENDReserve()) {
				ret += " (uses END Reserve)";
			} else {
				ret += " (uses Personal END)";
			}
		}
		return ret;
	}

	@Override
	public GenericDialog getDialog(boolean isNew, boolean isPower) {
		CombatSenseDialog dialog = new CombatSenseDialog(this, isNew, isPower);
		return dialog;
	}

	@Override
	public String getRoll() {
		String ret = "";
		if (HeroDesigner.getActiveHero() != null) {
			Characteristic charac = HeroDesigner.getActiveHero()
					.getCharacteristic(characteristic);
			int roll = (int) Rounder.roundHalfUp(11 + getLevels()
					* getLevelValue());
			int roll2 = (int) Rounder.roundHalfUp(11 + getLevels()
					* getLevelValue());
			if (getLevels() < 0) {
				roll = getMinimumLevel();
			} else if ((charac != null) && !charac.getXMLID().equals("GENERAL")) {
				roll = 9 + (int) (Rounder
						.roundHalfUp(charac.getPrimaryValue() / 5d) + getLevels()
						* getLevelValue());
				roll2 = 9 + (int) (Rounder.roundHalfUp(charac
						.getSecondaryValue() / 5d) + getLevels()
						* getLevelValue());
			} else if ((charac != null) && charac.getXMLID().equals("GENERAL")) {
				roll = 9 + (int) (Rounder.roundHalfUp(HeroDesigner
						.getActiveTemplate().getGeneralLevel() / 5d) + getLevels()
						* getLevelValue());
			}
			ret += roll + "-";
			if (roll != roll2) {
				ret += "/" + roll2 + "-";
			}
		} else {
			ret = "11-";
		}
		return ret;
	}

	@Override
	public Element getSaveXML() {
		Element root = super.getSaveXML();
		root.setAttribute("CHARACTERISTIC", Constants
				.getCharacteristicString(getCharacteristic()));
		root.setAttribute("LEVELS", "" + getLevels());
		return root;
	}

	@Override
	protected void init(Element element) {
		display = "Combat Sense";
		baseCost = 15;
		levelCost = 1;
		levelValue = 1;
		exclusive = true;
		characteristicChoices = new ArrayList<CharacteristicChoice>();
		characteristic = Constants.GENERAL;
		super.init(element);
		Element charChoice = element.getChild("CHARACTERISTIC_CHOICE");
		if (charChoice != null) {
			java.util.List choices = charChoice.getChildren("ITEM");
			Iterator iterator = choices.iterator();
			while (iterator.hasNext()) {
				Element item = (Element) iterator.next();
				if (item == null) {
					continue;
				}
				CharacteristicChoice charCh = new CharacteristicChoice(item);
				characteristicChoices.add(charCh);
			}
		}
	}

	@Override
	public void restoreFromSave(Element root) {
		String check = XMLUtility.getValue(root, "CHARACTERISTIC");
		if ((check != null) && (check.trim().length() > 0)) {
			setCharacteristic(Constants.getCharacteristicInteger(check));
		}
		super.restoreFromSave(root);
	}

	/**
	 * Sets the Characteristic to base the roll off of.
	 * 
	 * @see com.hero.util.Characteristics
	 * @param val
	 */
	public void setCharacteristic(int val) {
		for (CharacteristicChoice choice : getCharacteristicChoices()) {
			if (choice.getCharacteristic() == val) {
				characteristic = val;
				if (choice.getBaseCost() >= 0) {
					setBaseCost(choice.getBaseCost());
				}
				if (choice.getLevelCost() >= 0) {
					setLevelCost(choice.getLevelCost());
				}
				if (choice.getLevelValue() >= 0) {
					setLevelValue(choice.getLevelValue());
				}
				if (choice.getMinimumCost() > -999) {
					setMinimumCost(choice.getMinimumCost());
				}
				if (choice.getMinimumLevel() >= 0) {
					setMinimumLevel(choice.getMinimumLevel());
				}
			}
		}
	}

}